/****************************************************************
 * xitalkbiff.cc : xitalkbiff Main File
 *
 * (c) 1995 GANA.
 * (c) 1995-1998 Yutaka Oiwa <oiwa@is.s.u-tokyo.ac.jp>.
 *
 * You may distribute this file under the terms of license
 * agreement specified in the files README and LICENCE.
 *
 * $Id: xitalkbiff.cc,v 1.45 2005/06/01 13:57:55 yutaka Exp $
 ****************************************************************/

#define VERSION "1.5.4"

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <ctype.h>
#include <stdarg.h>
#include "Defines.h"
#include "Monkey.h"
#include "ItalkServer.h"
#include "ItalkPipeConnection.h"

#ifdef DEBUG
int Debug_Level;

void Dprintf(int d, char *p, ...)
{
  va_list ap;
  if(d & Debug_Level) {
    va_start(ap, p);
    vfprintf(stderr, p, ap);
    va_end(ap);
  }
}
void Dperror(int d, char *p)
{
  if (d & Debug_Level) {
    perror(p);
  }
}
#endif

static String default_resources[]=
{
  "*Title: XItalkBiff",
  "*star: Yellow",
  NULL,
};

static XrmOptionDescRec options[]=
{
{"--help",   ".help",          XrmoptionNoArg,  (XPointer)"True"},
{"-help",    ".help",          XrmoptionNoArg,  (XPointer)"True"},
{"-h",       ".help",          XrmoptionNoArg,  (XPointer)"True"},
{"-add",     ".addServer",     XrmoptionNoArg,  (XPointer)"True"},
{"-l",       ".landscape",     XrmoptionNoArg,  (XPointer)"True"},
{"-nl",      ".landscape",     XrmoptionNoArg,  (XPointer)"False"},
{"-fn",      ".font",          XrmoptionSepArg, (XPointer)NULL},
{"-fn8",     ".font",          XrmoptionSepArg, (XPointer)NULL},
{"-fnAscii", ".font",          XrmoptionSepArg, (XPointer)NULL},
{"-fn16",    ".kanjiFont",     XrmoptionSepArg, (XPointer)NULL},
{"-fk",      ".kanjiFont",     XrmoptionSepArg, (XPointer)NULL},
{"-fnKanji", ".kanjiFont",     XrmoptionSepArg, (XPointer)NULL},
{"-fr",      ".romanKanaFont", XrmoptionSepArg, (XPointer)NULL},
{"-fnKana",  ".romanKanaFont", XrmoptionSepArg, (XPointer)NULL},
{"-star",    ".star",          XrmoptionSepArg, (XPointer)NULL},
{"-fg",      ".foreground",    XrmoptionSepArg, (XPointer)NULL},
{"-bg",      ".background",    XrmoptionSepArg, (XPointer)NULL},
{"-fg2",     ".foreground2",   XrmoptionSepArg, (XPointer)NULL},
{"-bg2",     ".background2",   XrmoptionSepArg, (XPointer)NULL},
{"-p",       ".popup",         XrmoptionNoArg,  (XPointer)"True"},
{"-np",      ".popup",         XrmoptionNoArg,  (XPointer)"False"},
{"-pt",      ".popupTime",     XrmoptionSepArg, (XPointer)NULL},
{"-n",       ".showNumber",    XrmoptionNoArg,  (XPointer)"True"},
{"-nn",      ".showNumber",    XrmoptionNoArg,  (XPointer)"False"},
{"-b",       ".beep",          XrmoptionNoArg,  (XPointer)"True"},
{"-nb",      ".beep",          XrmoptionNoArg,  (XPointer)"False"},
{"-format",  ".format",        XrmoptionSepArg, (XPointer)NULL},
#ifdef DEBUG
{"-debug",   ".debug",         XrmoptionSepArg, (XPointer)NULL},
#endif
};

static struct AppData
{
  String italkpipe;
  XFontStruct *fsAscii;
  XFontStruct *fsKanji;
  XFontStruct *fsKana;
  Pixel foreground;
  Pixel foreground2;
  Pixel star;
  Pixel background;
  Pixel background2;
  Bool   help;
  Bool   landscape;
  int    format;
  Bool   popup;
  int    popup_time;
  Dimension min_width;
  Dimension max_width;
  Bool   show_number;
  Bool   beep;
  Pixel  color[8];
  int    wait_time;
  int    keepalive_time;
  Bool   add_server;
  int    gravity;
  int    buggy_border;
#ifdef DEBUG
  int    debug_level;
#endif
  int    bell_pitch;
} app_data;

static XtResource resources[]=
{
{ "italkServer",  "ItalkServer",   XtRString,  sizeof(String),
  XtOffset(AppData*, italkpipe),   XtRString,
  (XtPointer)"main.italk.ne.jp:12345"},
{ "font",         "Font",          XtRFontStruct,  sizeof(XFontStruct *),
  XtOffset(AppData*, fsAscii),   XtRString,
  (XtPointer)"-*-times-bold-r-*-*-14-*-*-*-*-*-iso8859-1"}, 
{ "romanKanaFont","RomanKanaFont", XtRFontStruct,  sizeof(XFontStruct *),
  XtOffset(AppData*, fsKana),    XtRString,
  (XtPointer)"-*-fixed-*-*-*-*-14-*-*-*-*-*-jisx0201.1976-0"},
{ "kanjiFont",    "KanjiFont",     XtRFontStruct,  sizeof(XFontStruct *),
  XtOffset(AppData*, fsKanji),   XtRString,
  (XtPointer)"-*-fixed-*-*-*-*-14-*-*-*-*-*-jisx0208.1983-0"},
{ "foreground",   "Foreground",    XtRPixel,  sizeof(Pixel),
  XtOffset(AppData*, foreground),  XtRString,    (XtPointer)"Black"},
{ "foreground2",  "Foreground",    XtRPixel,  sizeof(Pixel),
  XtOffset(AppData*, foreground2), XtRString,    (XtPointer)"Black"},
{ "star",         "Foreground",    XtRPixel,  sizeof(Pixel),
  XtOffset(AppData*, star),        XtRString,    (XtPointer)"Yellow"},
{ "background",   "Background",    XtRPixel,  sizeof(Pixel),
  XtOffset(AppData*, background),  XtRString,    (XtPointer)"gray"},
{ "background2",  "Background",    XtRPixel,  sizeof(Pixel),
  XtOffset(AppData*, background2), XtRString,    (XtPointer)"gray60"},
{ "gravity",      "Gravity",       XtRGravity, sizeof(int),
  XtOffset(AppData*, gravity),     XtRString,    (XtPointer)"NorthWest"},
{ "buggyBorder",  "buggyBorder",   XtRBoolean, sizeof(Boolean),
  XtOffset(AppData*, buggy_border),XtRImmediate, (XtPointer)False},
{ "help",         "Help",          XtRBoolean, sizeof(Boolean),
  XtOffset(AppData*, help),        XtRImmediate, (XtPointer)False},
{ "addServer",    "AddServer",     XtRBoolean, sizeof(Boolean),
  XtOffset(AppData*, add_server),  XtRImmediate, (XtPointer)False},
{ "landscape",    "Landscape",     XtRBoolean, sizeof(Boolean),
  XtOffset(AppData*, landscape),   XtRImmediate, (XtPointer)False},
{ "popup",        "Popup",         XtRBoolean, sizeof(Boolean),
  XtOffset(AppData*, popup),       XtRImmediate, (XtPointer)False},
{ "popupTime",    "PopupTime",     XtRInt,     sizeof(int),
  XtOffset(AppData*, popup_time),  XtRImmediate, (XtPointer)10},
{ "retryInterval","Interval",      XtRInt,     sizeof(int),
  XtOffset(AppData*, wait_time),   XtRImmediate, (XtPointer)60},
{ "keepAlive",    "KeepAlive",     XtRInt,     sizeof(int),
  XtOffset(AppData*, keepalive_time), XtRImmediate, (XtPointer)60},
{ "minimumWidth", "MinimumWidth",  XtRDimension, sizeof(Dimension),
  XtOffset(AppData*, min_width),   XtRImmediate, (XtPointer)20},
{ "maximumWidth", "MaximumWidth",  XtRDimension, sizeof(Dimension),
  XtOffset(AppData*, max_width),   XtRImmediate, (XtPointer)300},
{ "format",       "Format",        XtRInt,     sizeof(int),
  XtOffset(AppData*, format),      XtRImmediate, (XtPointer)0},
{ "showNumber",   "ShowNumber",    XtRBoolean, sizeof(Boolean),
  XtOffset(AppData*, show_number), XtRImmediate, (XtPointer)True},
{ "beep",         "Beep",          XtRBoolean, sizeof(Boolean),
  XtOffset(AppData*, beep),        XtRImmediate, (XtPointer)True},
{ "color0",       "Color0",        XtRPixel,   sizeof(Pixel),
  XtOffset(AppData*, color[0]),    XtRString,    (XtPointer)"Black"},
{ "color1",       "Color1",        XtRPixel,   sizeof(Pixel),
  XtOffset(AppData*, color[1]),    XtRString,    (XtPointer)"Red"},
{ "color2",       "Color2",        XtRPixel,   sizeof(Pixel),
  XtOffset(AppData*, color[2]),    XtRString,    (XtPointer)"Green"},
{ "color3",       "Color3",        XtRPixel,   sizeof(Pixel),
  XtOffset(AppData*, color[3]),    XtRString,    (XtPointer)"Yellow"},
{ "color4",       "Color4",        XtRPixel,   sizeof(Pixel),
  XtOffset(AppData*, color[4]),    XtRString,    (XtPointer)"Blue"},
{ "color5",       "Color5",        XtRPixel,   sizeof(Pixel),
  XtOffset(AppData*, color[5]),    XtRString,    (XtPointer)"Magenta"},
{ "color6",       "Color6",        XtRPixel,   sizeof(Pixel),
  XtOffset(AppData*, color[6]),    XtRString,    (XtPointer)"Cyan"},
{ "color7",       "Color7",        XtRPixel,   sizeof(Pixel),
  XtOffset(AppData*, color[7]),    XtRString,    (XtPointer)"White"},
{ "bellPitch",    "BellPitch",     XtRInt,     sizeof(int),
  XtOffset(AppData*, bell_pitch),  XtRImmediate, (XtPointer)0},
#ifdef DEBUG
{ "debug",        "Debug",         XtRInt,     sizeof(int),
  XtOffset(AppData*, debug_level), XtRImmediate, (XtPointer)0},
#endif
};

////////////////////////////////////////////////////////////////
// Font Drawing

int GetFontHeight(XFontStruct *flist[])
{
  int y = 0, max = 0;

  for(; *flist; flist++) {
    y = (*flist)->max_bounds.ascent 
      + (*flist)->max_bounds.descent;
    if (max < y)
      max = y;
  }
  return max;
}

enum CharType { Null, Ascii, Kana, Kanji, 
		Star, Color, OverwriteStart, 
		OverwriteEnd, InvalidChar };

const char *ParseString(const char *c, int &char_r, 
			CharType &chartype_r, int &width_r, 
			int &fg, int &bg)
{
  int dummy;
  XCharStruct cs;

  char_r=0;
  chartype_r=InvalidChar;
  width_r=0;

#define returnNULL do { chartype_r=Null; return NULL; } while (0)

  if (*c==0) returnNULL;

  else if (*c=='\x1b')
  {
    c++;

    if (*c=='\0') returnNULL;

    else if (*c=='[')
    {
      c++;
      if (*c=='m')
      {
        fg=-1;
        bg=-2;
        chartype_r=Color;
        return c+1;
      }
      int n=0;
      while (1)
      {
        if ('0'<=*c && *c<='9')
        {
          n=n*10+*c-'0';
          c++;
        }
        else if (*c==';' || *c=='m')
        {
          if      (30<=n && n<=37) fg=n;
          else if (40<=n && n<=47) bg=n;
          else if (n==7)
          {
            int tmp;
            tmp=fg;
            fg=bg;
            bg=tmp;
          }
          n=0;
          if (*c=='m')
          {
            chartype_r=Color;
            return c+1;
          }
          c++;
        }
        else return c;
      }
    }
    else if (*c=='0')
    {
      chartype_r=OverwriteStart;
      return c+1;
    }
    else if (*c=='1')
    {
      chartype_r=OverwriteEnd;
      return c+1;
    }
  }

  else if ((*c & 0x80) == 0)
  {
    char_r=*c;
    chartype_r=Ascii;

    XTextExtents(app_data.fsAscii, c, 1, &dummy, &dummy, &dummy, &cs);
    width_r=cs.width;

    return c+1;
  }

  else if (*c=='\x8e')
  {
    c++;
    if (!*c) returnNULL;

    char_r=*c;
    chartype_r=Kana;

    char k[1];
    *k=char_r;
    XTextExtents(app_data.fsKana, k, 1, &dummy, &dummy, &dummy, &cs);
    width_r=cs.width;

    return c+1;
  }

  else if ((c[0]=='\xa1' && c[1]=='\xfa') ||
           (c[0]=='\xa1' && c[1]=='\xf9'))
    chartype_r=Star;
  else
    chartype_r=Kanji;

  if (!*(c+1)) returnNULL;

  char k[2];
  k[0]=c[0]&0x7f;
  k[1]=c[1]&0x7f;
  XTextExtents16(app_data.fsKanji, (XChar2b*)k, 1, &dummy, &dummy, &dummy, &cs);
  width_r=cs.width;

  char_r=(k[0]<<8)|k[1];

  return c+2;
}

void DrawChar(Display *d, Pixmap pxm, GC pxmgc, 
	      int x, int y, int char_, CharType chartype, int f)
{
  if (chartype==Ascii)
  {
    char k[1];
    *k=char_;
    XSetFont(d, pxmgc, app_data.fsAscii->fid);
    if (f) XDrawImageString(d, pxm, pxmgc, x, y, k, 1);
    else   XDrawString     (d, pxm, pxmgc, x, y, k, 1);
  }
  else if (chartype==Kana)
  {
    char k[1];
    *k=char_;
    XSetFont(d, pxmgc, app_data.fsKana->fid);
    if (f) XDrawImageString(d, pxm, pxmgc, x, y, k, 1);
    else   XDrawString     (d, pxm, pxmgc, x, y, k, 1);
  }
  else if (chartype==Kanji || chartype==Star)
  {
    char k[2];
    k[0]=char_>>8;
    k[1]=char_&0x7f;
    XSetFont(d, pxmgc, app_data.fsKanji->fid);
    if (f) XDrawImageString16(d, pxm, pxmgc, x, y, (XChar2b*)k, 1);
    else   XDrawString16     (d, pxm, pxmgc, x, y, (XChar2b*)k, 1);
  }
}


static int DrawText(Display *d, Pixmap pxm, GC pxmgc, 
		    int x, int y, int max_width, const char *str, unsigned long _fg, unsigned long _bg)
{
  int esc_fg=-1;
  int esc_bg=-2;
  unsigned long fg=_fg;
  unsigned long bg=_bg;

  if (x>=0 && y>=0)
  {
    XSetForeground(d, pxmgc, fg);
    XSetBackground(d, pxmgc, bg);
    if (max_width > 0) {
      static XRectangle rect = {0,0,0,0};
      rect.width = max_width;
      rect.height = 65535;
      XSetClipRectangles(d, pxmgc, x, y, &rect, 1, YXSorted);
    }
  }

  y+=max(max(app_data.fsAscii->max_bounds.ascent,
             app_data.fsKanji->max_bounds.ascent),
         app_data.fsKana->max_bounds.ascent);

  int width=0;
  int overwrite_width=0;
  int if_overwrite=0;
  
  while (1)
  {
    int char_;
    CharType chartype_;
    int width_;
    str=ParseString(str, char_, chartype_, width_, esc_fg, esc_bg);

    switch (chartype_)
    {
    case Null:
      if (x>=0 && y>=0 && max_width > 0)
	XSetClipMask(d, pxmgc, None);
      return width;
    case Star:
      if (x>=0 && y>=0)
        XSetForeground(d, pxmgc, app_data.star);
    case Ascii:
    case Kana:
    case Kanji:
      if (x>=0 && y>=0)
        DrawChar(d, pxm, pxmgc, x+width, y, char_, chartype_, (if_overwrite<=1));
      if (if_overwrite)
      {
        if_overwrite=2;
        overwrite_width=max(overwrite_width, width_);
      }
      else width+=width_;
      if (x>=0 && y>=0)
        XSetForeground(d, pxmgc, fg);
      break;
    case Color:
      if (x>=0 && y>=0)
      {
        if (esc_fg>0) XSetForeground(d, pxmgc, fg=app_data.color[esc_fg%10%8]);
        else if (esc_fg==-1) XSetForeground(d, pxmgc, fg=_fg);
        else if (esc_fg==-2) XSetForeground(d, pxmgc, fg=_bg);
        if (esc_bg>0) XSetBackground(d, pxmgc, bg=app_data.color[esc_bg%10%8]);
        else if (esc_bg==-1) XSetBackground(d, pxmgc, bg=_fg);
        else if (esc_bg==-2) XSetBackground(d, pxmgc, bg=_bg);
      }
      break;
    case OverwriteStart:
      overwrite_width=0;
      if_overwrite=1;
      break;
    case OverwriteEnd:
      width+=overwrite_width;
      overwrite_width=0;
      if_overwrite=0;
      break;
    case InvalidChar:
      break;
    }
  }
}

static int GetWidth(Display *d, Window pxm, GC pxmgc, 
		    const char *str)
{
  return DrawText(d, pxm, pxmgc, -1, -1, -1, str, 0, 0);
}

////////////////////////////////////////////////////////////////
// GUI Class
////////////////////////////////////////////////////////////////

// GUI Entry

char *NewString(const char *s)
{
  char *p = new char [strlen(s) + 1];
  if (!p) abort();
  strcpy(p, s);
  return p;
}

class GUI_Entry {
public:
  enum { MONKEY, SERVER, MESSAGE } flag;
  char *name;
  char *where;
  char *status; 
  char *message;
  int num;

  GUI_Entry(const Monkey &p) {
    flag = MONKEY;
    name = NewString(p.who);
    where = NewString(p.where);
    status = NewString(p.status);
    message = NULL;
    num = p.no;
  }

  GUI_Entry(const ItalkServer &p) {
    flag = SERVER;
    name = NewString(p.Get_Name());
    where = NewString(p.Get_FullName());
    status = NULL;
    num = p.Get_NumMonkeys();
    if (! p.GetMessage())
      message = NULL;
    else {
      message = NewString(p.GetMessage());
    }
  }
  
  GUI_Entry(const char *p) {
    flag = MESSAGE;
    name = NewString(p);
    where = NULL;
    status = NULL;
    message = NULL;
    num = -1;
  }

  ~GUI_Entry() {
    if (name)
      delete [] name;
    if (where)
      delete [] where;
    if (status)
      delete [] status;
    if (message)
      delete [] message;
  }
  
  const char *username() {
    return name;
  }

  const char *num_str(int l, int f, int n);

  bool reverse() {return flag == SERVER;}
  const char *dispstr(int l, int f, int n);
  const char *messagestr(int l, int f, int n);
};

const char *GUI_Entry::num_str(int l, int f, int s)
{
  static char buf[256];
  switch(flag) {
  case MONKEY:
    if (l && s) {
      switch (f) {
      case 1: 
	{
	  int l = strlen(where);
	  char *s = strchr(where, '.');
	  if (s) l = s - where;
	  if (l > 254) l = 254;
	  sprintf(buf, "@%.*s", l, where);
	  return buf;
	}
      case 2:
	if (status && status[0]) {
	  sprintf(buf, "<%.253s>", status);
	  return buf;
	}
	return "";
      default:
	; // fall through
      }
    }
    sprintf(buf, "%d", num);
    return buf;

  case SERVER:
    if (l && s) {
      if (message) {
	sprintf(buf, "[%.253s]", message);
	return buf;
      }
    } else {
      if (message || num <= 0)
	return "";
    }
    sprintf(buf, "(%d)", num);
    return buf;

  case MESSAGE:
  default:
    return "";
  }
}

const char *GUI_Entry::messagestr(int l, int, int)
{
  if (flag != SERVER || l)
    return NULL;
  if (message)
    return message;
  if (num == 0 && !l)
    return "No Monkeys.";
  return NULL;
}

const char *GUI_Entry::dispstr(int l, int f, int s) {
  static char buf[BUFSIZ];
  switch(flag) {
  case MONKEY:
    if (l)
      return name;
    switch(f) {
    case 1:
      sprintf(buf, "%.253s@%.253s", name, where);
      return buf;
    case 2:
      if (status && status[0]) {
	sprintf(buf, "%.253s <%.253s>", name, status);
	return buf;
      } /* else fallthrough */
    default:
      return name;
    }
    break;
  case SERVER:
    if (l) {
      if (s || !message) {
	return name;
      }
      sprintf(buf, "%.253s [%.253s]", name, message);
      return buf;
    } else {
      switch(f) {
      case 1:
	return where;
      default:
	return name;
      }
    }
    break;
  case MESSAGE:
  default:
    return name;
  }
}

////////////////////////////////////////////////////////////////
// Biff Window

class BiffWindow {
private:
  XtAppContext app_con;
  static const int MAX_SVRS = 10;
  static const int MAX_ENTRIES = 400; // MAX_ENTRIES >= MAX_SVRS * (MAX_MONKEYS + 1)
  Widget widget;
  Window w, root;
  GC gc, rootgc;
  Pixmap pxm;
  GC pxmgc;
  int font_height;
  ItalkPipeConnection *connections[MAX_SVRS];
  int n_server;
  GUI_Entry *entries[MAX_ENTRIES];
  XtWorkProcId block_id;
  XtIntervalId popdown_id;

  Display *d;
  unsigned int height, width;
  unsigned int r_height, r_width;
  int repaint_flag;

  void acquire_data();
  friend Boolean X_repaint_hook(XtPointer);
  void repaint_hook();

  friend void cb_X_popdown(void *, XtIntervalId *);
  void popdown();
  void popup();

  void draw_entry(int draw, GUI_Entry *who, int type, int *p_revp, int *ip, int *xp, int *yp);

public:
  BiffWindow(XtAppContext app_con_, Widget widget_);
  ~BiffWindow();
  void register_connection(ItalkPipeConnection *);
  inline void redraw();
  inline void calc_size();
  inline unsigned int get_height() { return height; }
  inline unsigned int get_width() { return width; }
  void draw_or_calc(int, int = 1);
  void register_repaint();
  void expose(XEvent *);
  void bell();
};

BiffWindow::BiffWindow(XtAppContext app_con_, Widget widget_)
{
  app_con = app_con_;
  widget = widget_;
  w = 0; root = 0;
  gc = 0; rootgc = 0;
  pxm = 0; pxmgc = 0;
  d = 0;
  height = 0; width = 0;
  r_height = 1; r_width = 1;
  n_server = 0;
  repaint_flag = 0;
  block_id = 0;
  popdown_id = 0;

  for(int i = 0; i < MAX_SVRS; i++)
    connections[i] = NULL;
  for(int i = 0; i < MAX_ENTRIES; i++)
    entries[i] = NULL;

  {
    XFontStruct *farray[4];
    farray[0] = app_data.fsAscii;
    farray[1] = app_data.fsKanji;
    farray[2] = app_data.fsKana;
    farray[3] = NULL;
    font_height = GetFontHeight(farray);
  }

  if (app_data.popup) 
    popup();
}

BiffWindow::~BiffWindow()
{
  for(int i = 0; i < n_server; i++)
    delete connections[i];
}

void BiffWindow::register_connection(ItalkPipeConnection *conn)
{
  connections[n_server] = conn;
  n_server++;
}

void BiffWindow::register_repaint()
{
  Dprintf(4, "Repaint Registered: old=%d\n", repaint_flag);
  if (repaint_flag == 0) {
    block_id = XtAppAddWorkProc(app_con, X_repaint_hook, (XtPointer)this);
    Dprintf(4, "Hook Registered: %d\n", block_id);
    repaint_flag = 1;
  }
}

void BiffWindow::acquire_data(void)
{
  for(GUI_Entry **t=entries; *t; t++) {
    delete *t;
  }

  GUI_Entry **ent = entries;
  
  for(int i = 0; i < n_server; i++) {
    ItalkServer *svr = connections[i]->Server();
    *ent++ = new GUI_Entry(*svr);
    const char *message = svr->GetMessage();
    if(message) {
      // If Message Exist, No Data is Displayed.
    } else {
      const Monkey *const *j = svr->GetMonkeys();
      for(; *j; j++) {
	*ent++ = new GUI_Entry(**j);
      }
    }
  }
  *ent = NULL;
}

void BiffWindow::popup()
{
  if (!app_data.popup)
    return;

  Dprintf(4, "popup()\n");
  if (popdown_id) {
    Dprintf(4, "removed old timeout()\n");
    XtRemoveTimeOut(popdown_id);
  }
  XtVaSetValues(widget, XtNiconic, false, 0);
  popdown_id = XtAppAddTimeOut(app_con, app_data.popup_time * 1000, cb_X_popdown, (void *)this);
}

void BiffWindow::popdown()
{
  Dprintf(4, "popdown()\n");

  popdown_id = 0;

  XtVaSetValues(widget, XtNiconic, true, 0);
}  

void BiffWindow::bell()
{
  XKeyboardState orig;
  XKeyboardControl con;
  if (!d) d = XtDisplay(widget);

  if (app_data.beep) {
    if (app_data.bell_pitch) {
      XGetKeyboardControl(d, &orig);
      con.bell_pitch = app_data.bell_pitch;
      XChangeKeyboardControl(d, KBBellPitch, &con);
    }
    XBell(d, 0);
    if (app_data.bell_pitch) {
      con.bell_pitch = orig.bell_pitch;
      XChangeKeyboardControl(d, KBBellPitch, &con);
    }
  }
  popup();
}

void BiffWindow::draw_or_calc(int draw, int get)
{
  Dprintf(4, "Draw|Calc (%d,%d)\n", draw, get);
  if (!d) d = XtDisplay(widget);
  if (!root) {
    root = DefaultRootWindow(d);
    rootgc=XCreateGC(d, root, 0, 0);
  }
  if (draw) {
    if (!w) w = XtWindow(widget);
    if (w) {
      if (!gc) gc = XCreateGC(d, root, 0, 0);
      if (pxm)   XFreePixmap(d, pxm);
      if (pxmgc) XFreeGC(d, pxmgc);
      draw_or_calc(0,get);
      get = 0;
      pxm=XCreatePixmap(d, w, width, height, DefaultDepth(d, 0));
      pxmgc=XCreateGC(d, pxm, 0, 0);
    } else 
      draw = 0; // Not Yet Realized
  }
  if (get)
    acquire_data();

  int x = 0, y = 0, i = 0;
  
  int p_rev = 0;
  for(GUI_Entry **p = entries; *p; p++) {
    GUI_Entry *who = *p;
    draw_entry(draw, who, 0, &p_rev, &i, &x, &y);
    draw_entry(draw, who, 1, &p_rev, &i, &x, &y);
  }

  if (!draw) {
    if (app_data.landscape) {
      height = font_height + 12 + ((app_data.show_number) ? 2 + font_height : 0);
      width = x;
    } else {
      height = y;
      width = x + 10;
      if (width < app_data.min_width) width = app_data.min_width;
      if (width > app_data.max_width) width = app_data.max_width;
      if (width < 20) width = 20;
    }
    if (width < 1) width = 1;
    if (height < 1) height = 1;
  }
}

void BiffWindow::draw_entry(int draw, GUI_Entry *who, int type, 
			     int *p_revp, int *ip, int *xp, int *yp)
{
  int &i = *ip, &p_rev = *p_revp; // alias
  int &x = *xp, &y = *yp; // alias
  const char *s1 = NULL, *s2 = NULL;
  int rev;
  
  if (type) {
    s1 = who->messagestr(app_data.landscape, app_data.format, app_data.show_number);
    s2 = "";
    rev = 0;
  } else {
    s1 = who->dispstr(app_data.landscape, app_data.format, app_data.show_number);
    s2 = who->num_str(app_data.landscape, app_data.format, app_data.show_number);
    rev = who->reverse();
  }
//  Dprintf(4, "draw_entry <%s>,<%s>, i%d r%d-%d\n", s1 ? s1 : "(null)",
//	  s2 ? s2 : "(null)",i, rev, p_rev);
  if (!s1) return;

  Pixel back = i ? app_data.background : app_data.background2;
  Pixel back2 = i ? app_data.background2 : app_data.background;
  Pixel fore = app_data.foreground;
  Pixel fore2 = app_data.foreground2;
  if (rev) {
    back = back2 = app_data.foreground;
    fore = fore2 = app_data.background;
    i = 1; // next 1
  } else {
    i = !i;
  }
  int sep_needed = (rev == p_rev && back == back2);
  
  if (app_data.landscape) {
    if (sep_needed) {
      if (draw) {
	XSetForeground(d, pxmgc, fore);
	XDrawLine(d, pxm, pxmgc, x, 0, x, height);
      }
      x++;
    }
    int w1 = GetWidth(d, root, rootgc, s1);
    int w2 = 0;
    if (app_data.show_number) {
      w2 =GetWidth(d, root, rootgc, s2);
    }
    int wx = max(w1,w2);
    if (draw) {
      int xx=x+3+(wx-w1)/2;
      XSetForeground(d, pxmgc, back);
      XFillRectangle(d, pxm, pxmgc, x, 0, wx+6, height);
      DrawText(d, pxm, pxmgc, xx, 6, -1, s1, fore, back);
      if (app_data.show_number) {
	int xx=x+3+(wx-w2)/2;
	DrawText(d, pxm, pxmgc, xx, 6+2+font_height, -1, s2,
		 fore2, back);
      }
    }
    x += wx + 6;
  } else { // portrait
    if (sep_needed) {
      if (draw) {
	XSetForeground(d, pxmgc, fore);
	XDrawLine(d, pxm, pxmgc, 0, y, width, y);
      }
      y++;
    }
    if (draw) {
      XSetForeground(d, pxmgc, back);
      XFillRectangle(d, pxm, pxmgc, 0, y, width, font_height+2);
    }
    int s1_w = GetWidth(d, root, rootgc, s1);
    int ww = s1_w + 2;
    if (app_data.show_number) {
      int w_max = width - 10;
      int s2_w = GetWidth(d, root, rootgc, s2);
      if (s2_w) {
	ww += 4 + s2_w;
	w_max -= 4 + s2_w;
      }
      if (draw) {
	if (w_max >= s1_w)
	  w_max = -1; // suppress useless clipping
	else
	  Dprintf(4, "clip w_max=%d, textwidth=%d, s1=%s\n", w_max, s1_w, s1);
	DrawText(d, pxm, pxmgc,       5,     y+1, w_max, s1, fore, back);
	DrawText(d, pxm, pxmgc, width-5-s2_w, y+1, -1, s2, fore2, back);
      }
    } else {
      if (draw) {
	int w_max = -1;
	int xpos = (int(width) - int(GetWidth(d, root, rootgc, s1)))/2;
	if (xpos < 5) {
	  xpos = 5;
	  w_max = width - 10;
	}
	DrawText(d, pxm, pxmgc, xpos, y+1, w_max, s1, fore, back);
      }
    }
    if (x < ww) x = ww;
    y += font_height+2;
  }
  p_rev = rev;
}

inline void BiffWindow::redraw() { draw_or_calc(1); }
inline void BiffWindow::calc_size() { draw_or_calc(0); }

inline void BiffWindow::repaint_hook() {
  Dprintf(4, "Repaint Hook Reached.\n");
  Dprintf(4, "Repaint Hook -Removed.\n");
  XtRemoveWorkProc(block_id);
  repaint_flag = 2;
  block_id = 0;
  expose(NULL);
}

static Atom         atom_wm_protocols, atom_wm_delete_window;

void BiffWindow::expose(XEvent *e)
{
  Dprintf(1, "Expose (%p,%d)...\n", e, repaint_flag);
  int f = 0;
  if (repaint_flag == 1) {
    Dprintf(1, "Repaint Hook Removed.\n");
    XtRemoveWorkProc(block_id);
    block_id = 0;
    f = 1;
  } else if (repaint_flag == 2) {
    Dprintf(1, "Called By Repaint_hook.\n");
    f = 1;
  }
  repaint_flag = 0;
  
  if (!w) w = XtWindow(widget);

  if (!w) {// realize
    Dprintf(1, "Realize.\n");
    XtVaSetValues(widget,
		  XtNwidth, width,
		  XtNheight, height,
		  XtNmaxWidth,  width,
		  XtNmaxHeight, height,
		  XtNminWidth,  width,
		  XtNminHeight, height,
		  NULL);
    XtRealizeWidget(widget);
    XtVaSetValues(widget, XtNinput, True, NULL);
    w = XtWindow(widget);

    atom_wm_protocols    =XInternAtom(d, "WM_PROTOCOLS",     False);
    atom_wm_delete_window=XInternAtom(d, "WM_DELETE_WINDOW", False);
    XSetWMProtocols(d, w, &atom_wm_delete_window, 1);
  }

  if (!w) return;
  if (!d) d = XtDisplay(widget);
  if (!gc) gc = XCreateGC(d, w, 0, 0);
  if (!pxm)
    redraw();
  if (width == r_width && height == r_height && e && !f) {
    XCopyArea(d, pxm, w, gc, 
              e->xexpose.x,     e->xexpose.y,
              e->xexpose.width, e->xexpose.height,
              e->xexpose.x,     e->xexpose.y);
    Dprintf(1, "done SIMPLE (%p).\n", e);
    return;
  }
  if (width != r_width || height != r_height) {
    Dimension x, y;
    Dimension oldw, oldh;
    Dimension border_w;
    XtVaGetValues(widget,
		  XtNx, &x,
		  XtNy, &y,
		  XtNwidth, &oldw,
		  XtNheight, &oldh,
		  XtNborderWidth, &border_w,
		  NULL);
    Dprintf(2, "Resize (%d,%d)[%d,%d(%d)]->(%dx%d).\n", r_width, r_height, 
	    oldw, oldh, border_w, width, height);
    int diff_w = (int)width - (int)oldw;
    int diff_h = (int)height - (int)oldh;

    Dimension newx = x, newy = y;
    int set = 0;
    switch (app_data.gravity) {
    case NorthEastGravity:
    case EastGravity:
    case SouthEastGravity:
      newx -= diff_w;
      if (app_data.buggy_border) {
	newx += border_w;
      }
      set |= 1;
    }
    switch (app_data.gravity) {
    case SouthWestGravity:
    case SouthGravity:
    case SouthEastGravity:
      newy -= diff_h;
      if (app_data.buggy_border) {
	newy += border_w;
      }
      set |= 2;
    }
    if (set) {
      Dprintf(2, "Move (%d,%d)->(%dx%d)[b%ds%d].\n", x, y, newx, newy, border_w, set);
//      XMoveResizeWindow(XtDisplay(widget), XtWindow(widget), x, y, width, height);
      XtVaSetValues(widget,
		    XtNx, newx, 
		    XtNy, newy,
		    XtNwidth, width,
		    XtNheight, height,
		    XtNmaxWidth,  width,
		    XtNmaxHeight, height,
		    XtNminWidth,  width,
		    XtNminHeight, height,
		    NULL);
    } else {
//      XResizeWindow(XtDisplay(widget), XtWindow(widget), width, height);
      XtVaSetValues(widget,
		    XtNwidth, width,
		    XtNheight, height,
		    XtNmaxWidth,  width,
		    XtNmaxHeight, height,
		    XtNminWidth,  width,
		    XtNminHeight, height,
		    NULL);
    }
    r_width = width; r_height = height;
  }
  XCopyArea(d, pxm, w, gc, 0, 0, width, height, 0, 0);
  Dprintf(1, "done FULL (%p).\n", e);
}

////////////////////////////////////////////////////////////////

static Widget       shell;
static Display      *d=NULL;
static XtAppContext acr;

int                 popup_timer=0;

BiffWindow *biff_win;

////////////////////////////////////////////////////////////////
// Xt Callbacks

Boolean X_repaint_hook(XtPointer s)
{
  ((BiffWindow *)s)->repaint_hook();
  return True;
}

void cb_X_popdown(void *s, XtIntervalId *)
{
  ((BiffWindow *)s)->popdown();
}

static void redraw_event(Widget , XEvent *e, String *, Cardinal *)
{
  biff_win->expose(e);
}

void cleanup()
{
  delete biff_win;
}

static void close_window_event(Widget, XEvent *e, String *, Cardinal *)
{
  if (e->type==ClientMessage &&
      e->xclient.message_type==atom_wm_protocols &&
      e->xclient.data.l[0]==(long)atom_wm_delete_window) {
    cleanup();
    exit(0);
  }
}
  
static void toggle_show_number_event(Widget, XEvent *, String *, Cardinal *)
{
  app_data.show_number=!app_data.show_number;
  biff_win->redraw();
  biff_win->register_repaint();
}

static void toggle_landscape_event(Widget, XEvent *, String *, Cardinal *)
{
  app_data.landscape=!app_data.landscape;
  biff_win->redraw();
  biff_win->register_repaint();
}

static void toggle_format_event(Widget, XEvent *, String *, Cardinal *)
{
  app_data.format++;
	app_data.format %= 3;
  biff_win->redraw();
  biff_win->register_repaint();
}

static void quit_event(Widget, XEvent *, String *, Cardinal *)
{
  cleanup();
  exit(0);
}

static XtActionsRec actions[]=
{
{"quit",               quit_event},
{"redraw",             redraw_event},
{"toggle_show_number", toggle_show_number_event},
{"toggle_landscape",   toggle_landscape_event},
{"close",              close_window_event},
{"toggle_format",      toggle_format_event},
};

static String translation=
"<ClientMessage>:   close()\n"
"<ConfigureNotify>: redraw()\n"
"<Key>s:            toggle_show_number()\n"
"<Key>n:            toggle_show_number()\n"
"<Key>t:            toggle_show_number()\n"
"<Btn1Down>:        toggle_show_number()\n"
"<Key>l:            toggle_landscape()\n"
"<Btn2Down>:        toggle_landscape()\n"
"<Key>f:            toggle_format()\n"
"<Btn3Down>:        toggle_format()\n"
"<Key>q:            quit()\n"
"<Key>Escape:       quit()\n"
"<Expose>:          redraw()\n";

void usage()
{
  fprintf(stderr,
	  "xitalkbiff " VERSION
#ifdef DEBUG
	  " [DEBUG]"
#endif
#ifdef OLD_COMPAT
	  " [OLD-COMPAT]"
#endif
#ifdef NO_ASYNC_CONNECT
	  " [NO-ASYNC-CONNECT]"
#endif
#ifdef ENABLE_ASYNC_DNS
	  " [ASYNC-DNS]"
#endif
	  "\n"
          "copyright (c) 1995-2000 GANA, Yutaka Oiwa\n"
	  "\n"
	  "usage: xitalkbiff [options ...] [host:port ...]\n"
	  "       -fn <ascii-font>, -fk <kanji-font>, -fr <kana-font>, \n"
	  "       -fg <color>, -fg2 <color>, -star <color>, -bg <color>, -bg2 <color>,\n"
	  "       -h, -l, -nl, -p, -pt <popuptime>, -n, -nn, -b, -nb, -iconic, "
#ifdef DEBUG
	  "\n"
	  "       -debug <level>, "
#endif
	  "etc.\n"
	  );
}

void newdata(int newp, void *ptr)
{
  Dprintf(8, "newdata(%d,%p)\n", newp, ptr);
  BiffWindow *self = (BiffWindow *)ptr;
  if (newp)
    self->bell();
  
  self->redraw();
  self->register_repaint();
}

int main(int argc, char *argv[])
{
  // X Initialize
  shell=XtVaAppInitialize
    (&acr, "XItalkBiff", options, XtNumber(options),
     &argc, argv, default_resources, NULL);
  d=XtDisplay(shell);
  XtAppAddActions(acr, actions, XtNumber(actions));
  XtGetApplicationResources(shell, XtPointer(&app_data), resources, 
			    XtNumber(resources), NULL, 0);
  XtAugmentTranslations(shell, XtParseTranslationTable(translation));

  if (app_data.help || (argv[1] && argv[1][0] == '-')) {
    usage();
    return 1;
  }

#ifdef DEBUG
  Debug_Level = app_data.debug_level;
#endif
  biff_win = new BiffWindow(acr, shell);

  if (argc == 1 || app_data.add_server) {
    char *str = NewString(app_data.italkpipe);
    char *q = str;
    while(1) {
      while(*q && isspace(*q)) q++; // skip '\n', '\t'
      if (!*q) break;
      char *r = q;
      while(*r && !isspace(*r)) r++;
      char tmp = *r;
      *r = '\0';
      Dprintf(8, "Server: >%s<\n", q);
      ItalkPipeConnection *p = 
	new ItalkPipeConnection(acr, q,
				app_data.wait_time, app_data.keepalive_time,
				newdata, biff_win);
      biff_win->register_connection(p);
      *r = tmp;
      q = r;
    }
    delete[] str;
  }
  for(int i = 1; i < argc; i++) {
    ItalkPipeConnection *p = 
      new ItalkPipeConnection(acr, argv[i], 
			      app_data.wait_time, app_data.keepalive_time, newdata, biff_win);
    biff_win->register_connection(p);
  }

  //  biff_winread_data(False);
  biff_win->draw_or_calc(0,1);
//  int width = biff_win->get_width();
//  int height = biff_win->get_height();
  XtVaSetValues
    (shell,
//     XtNwidth,     width,
//     XtNheight,    height,
//     XtNmaxWidth,  width,
//     XtNmaxHeight, height,
//     XtNminWidth,  width,
//     XtNminHeight, height,
     XtNinput,     True,
     NULL);

  // signal
  signal(SIGTERM, (void (*)(int))quit_event);
  signal(SIGINT, (void (*)(int))quit_event);

//  biff_win->expose(NULL);

  // main loop
  XtAppMainLoop(acr);
//  return 0;
}

