/****************************************************************
 *
 * async_dns_from_named.c : asyncronous dns lookup subroutines
 *                          derived from bind named resolver.
 *
 * edited (minced?) by Yutaka Oiwa at 1998-1999.
 *
 * Refer following notice (brought from original source)
 * for copyright and terms.
 *
 ****************************************************************/

/*
 * ++Copyright++ 1985, 1988, 1993
 * -
 * Copyright (c) 1985, 1988, 1993
 *    The Regents of the University of California.  All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * -
 * Portions Copyright (c) 1993 by Digital Equipment Corporation.
 * 
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies, and that
 * the name of Digital Equipment Corporation not be used in advertising or
 * publicity pertaining to distribution of the document or software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
 * CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 * -
 * --Copyright--
 */


#include <syslog.h>

typedef union {
    HEADER hdr;
    u_char buf[MAXPACKET];
} querybuf;

#define MAXALIASES      35
#define MAXADDRS        35

#undef res_dnok
#undef res_hnok

static struct hostent host;
static char hostbuf[8*1024];
static char *host_aliases[MAXALIASES];
static char *h_addr_ptrs[MAXADDRS + 1];
static const char AskedForGot[] =
                          "gethostby*.getanswer: asked for \"%s\", got \"%s\"";

typedef union {
    int32_t al;
    char ac;
} align;

#ifndef AF_INET6
#define AF_INET6     24
#endif
#ifndef INADDRSZ
#define INADDRSZ     4
#endif
#ifndef IN6ADDRSZ
#define IN6ADDRSZ    16
#endif
#ifndef NETDB_INTERNAL
#define NETDB_INTERNAL (-1)
#define NETDB_SUCCESS  0
#endif
#ifndef T_AAAA
#define T_AAAA          28
#endif
#ifndef HFIXEDSZ
#define HFIXEDSZ        12
#endif
#ifndef INT32SZ
#define INT32SZ         4
#define INT16SZ         2
#endif
#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN  64
#endif
#ifndef RES_USE_INET6
#define RES_USE_INET6   0x00002000
#endif
#ifndef _PATH_HOSTS
#define _PATH_HOSTS "/etc/hosts"
#endif

#define dprintf printf
#define PERIOD 0x2e
#define      hyphenchar(c) ((c) == 0x2d)
#define periodchar(c) ((c) == 0x2e) /* '.' */
#define asterchar(c) ((c) == 0x2a)
#define alphachar(c) (((c) >= 0x41 && (c) <= 0x5a) \
                   || ((c) >= 0x61 && (c) <= 0x7a))
#define digitchar(c) ((c) >= 0x30 && (c) <= 0x39)

#define borderchar(c) (alphachar(c) || digitchar(c))
#define middlechar(c) (borderchar(c) || hyphenchar(c))
#define domainchar(c) ((c) > 0x20 && (c) < 0x7f)

static int
res_dnok(dn)
        const char *dn;
{
        int ch;

        while ((ch = *dn++) != '\0')
                if (!domainchar(ch))
                        return (0);
        return (1);
}

static int
res_hnok(dn)
        const char *dn;
{
        int ppch = '\0', pch = PERIOD, ch = *dn++;

        while (ch != '\0') {
                int nch = *dn++;

                if (periodchar(ch)) {
                        NULL;
                } else if (periodchar(pch)) {
                        if (!borderchar(ch))
                                return (0);
                } else if (periodchar(nch) || nch == '\0') {
                        if (!borderchar(ch))
                                return (0);
                } else {
                        if (!middlechar(ch))
                                return (0);
                }
                ppch = pch, pch = ch, ch = nch;
        }
        return (1);
}

static void
map_v4v6_address(src, dst)
        const char *src;
        char *dst;
{
        u_char *p = (u_char *)dst;
        char tmp[INADDRSZ];
        int i;

        /* Stash a temporary copy so our caller can update in place. */
        bcopy(src, tmp, INADDRSZ);
        /* Mark this ipv6 addr as a mapped ipv4. */
        for (i = 0; i < 10; i++)
                *p++ = 0x00;
        *p++ = 0xff;
        *p++ = 0xff;
        /* Retrieve the saved copy and we're done. */
        bcopy(tmp, (void*)p, INADDRSZ);
}

static void
map_v4v6_hostent(hp, bpp, lenp)
        struct hostent *hp;
        char **bpp;
        int *lenp;
{
        char **ap;

        if (hp->h_addrtype != AF_INET || hp->h_length != INADDRSZ)
                return;
        hp->h_addrtype = AF_INET6;
        hp->h_length = IN6ADDRSZ;
        for (ap = hp->h_addr_list; *ap; ap++) {
                int i = sizeof(align) - ((u_long)*bpp % sizeof(align));

                if (*lenp < (i + IN6ADDRSZ)) {
                        /* Out of memory.  Truncate address list here.  XXX */
                        *ap = NULL;
                        return;
                }
                *bpp += i;
                *lenp -= i;
                map_v4v6_address(*ap, *bpp);
                *ap = *bpp;
                *bpp += IN6ADDRSZ;
                *lenp -= IN6ADDRSZ;
        }
}

static struct hostent *
getanswer(answer, anslen, qname, qtype)
        const querybuf *answer;
        int anslen;
        const char *qname;
        int qtype;
{
        register const HEADER *hp;
        register const u_char *cp;
        register int n;
        const u_char *eom;
        char *bp, **ap, **hap;
        int type, class, buflen, ancount, qdcount;
        int haveanswer, had_error;
        int toobig = 0;
        char tbuf[MAXDNAME];
        const char *tname;
        int (*name_ok) (const char *);

        tname = qname;
        host.h_name = NULL;
        eom = answer->buf + anslen;
        switch (qtype) {
        case T_A:
        case T_AAAA:
                name_ok = res_hnok;
                break;
        case T_PTR:
                name_ok = res_dnok;
                break;
        default:
                return (NULL);  /* XXX should be abort(); */
        }
        /*
         * find first satisfactory answer
         */
        hp = &answer->hdr;
        ancount = ntohs(hp->ancount);
        qdcount = ntohs(hp->qdcount);
        bp = hostbuf;
        buflen = sizeof hostbuf;
        cp = answer->buf + HFIXEDSZ;
        if (qdcount != 1) {
                h_errno = NO_RECOVERY;
                return (NULL);
        }
        n = dn_expand(answer->buf, eom, cp, bp, buflen);
        if ((n < 0) || !(*name_ok)(bp)) {
                h_errno = NO_RECOVERY;
                return (NULL);
        }
        cp += n + QFIXEDSZ;
        if (qtype == T_A || qtype == T_AAAA) {
                /* res_send() has already verified that the query name is the
                 * same as the one we sent; this just gets the expanded name
                 * (i.e., with the succeeding search-domain tacked on).
                 */
                n = strlen(bp) + 1;             /* for the \0 */
                if (n >= MAXHOSTNAMELEN) {
                        h_errno = NO_RECOVERY;
                        return (NULL);
                }
                host.h_name = bp;
                bp += n;
                buflen -= n;
                /* The qname can be abbreviated, but h_name is now absolute. */
                qname = host.h_name;
        }
        ap = host_aliases;
        *ap = NULL;
        host.h_aliases = host_aliases;
        hap = h_addr_ptrs;
        *hap = NULL;
        host.h_addr_list = h_addr_ptrs;
        haveanswer = 0;
        had_error = 0;
        while (ancount-- > 0 && cp < eom && !had_error) {
                n = dn_expand(answer->buf, eom, cp, bp, buflen);
                if ((n < 0) || !(*name_ok)(bp)) {
                        had_error++;
                        continue;
                }
                cp += n;                        /* name */
                type = _getshort(cp);
                cp += INT16SZ;                  /* type */
                class = _getshort(cp);
                cp += INT16SZ + INT32SZ;        /* class, TTL */
                n = _getshort(cp);
                cp += INT16SZ;                  /* len */
                if (class != C_IN) {
                        /* XXX - debug? syslog? */
                        cp += n;
                        continue;               /* XXX - had_error++ ? */
                }
                if ((qtype == T_A || qtype == T_AAAA) && type == T_CNAME) {
                        if (ap >= &host_aliases[MAXALIASES-1])
                                continue;
                        n = dn_expand(answer->buf, eom, cp, tbuf, sizeof tbuf);
                        if ((n < 0) || !(*name_ok)(tbuf)) {
                                had_error++;
                                continue;
                        }
                        cp += n;
                        /* Store alias. */
                        *ap++ = bp;
                        n = strlen(bp) + 1;     /* for the \0 */
                        if (n >= MAXHOSTNAMELEN) {
                                had_error++;
                                continue;
                        }
                        bp += n;
                        buflen -= n;
                        /* Get canonical name. */
                        n = strlen(tbuf) + 1;   /* for the \0 */
                        if (n > buflen || n >= MAXHOSTNAMELEN) {
                                had_error++;
                                continue;
                        }
                        strcpy(bp, tbuf);
                        host.h_name = bp;
                        bp += n;
                        buflen -= n;
                        continue;
                }
                if (qtype == T_PTR && type == T_CNAME) {
                        n = dn_expand(answer->buf, eom, cp, tbuf, sizeof tbuf);
                        if (n < 0 || !res_dnok(tbuf)) {
                                had_error++;
                                continue;
                        }
                        cp += n;
                        /* Get canonical name. */
                        n = strlen(tbuf) + 1;   /* for the \0 */
                        if (n > buflen || n >= MAXHOSTNAMELEN) {
                                had_error++;
                                continue;
                        }
                        strcpy(bp, tbuf);
                        tname = bp;
                        bp += n;
                        buflen -= n;
                        continue;
                }
                if (type != qtype) {
                        syslog(LOG_NOTICE|LOG_AUTH,
               "gethostby*.getanswer: asked for \"%s %s %s\", got type \"%s\"",
                               qname, p_class(C_IN), p_type(qtype),
                               p_type(type));
                        cp += n;
                        continue;               /* XXX - had_error++ ? */
                }
                switch (type) {
                case T_PTR:
                        if (strcasecmp(tname, bp) != 0) {
                                syslog(LOG_NOTICE|LOG_AUTH,
                                       AskedForGot, qname, bp);
                                cp += n;
                                continue;       /* XXX - had_error++ ? */
                        }
                        n = dn_expand(answer->buf, eom, cp, bp, buflen);
                        if ((n < 0) || !res_hnok(bp)) {
                                had_error++;
                                break;
                        }
#if MULTI_PTRS_ARE_ALIASES
                        cp += n;
                        if (!haveanswer)
                                host.h_name = bp;
                        else if (ap < &host_aliases[MAXALIASES-1])
                                *ap++ = bp;
                        else
                                n = -1;
                        if (n != -1) {
                                n = strlen(bp) + 1;     /* for the \0 */
                                if (n >= MAXHOSTNAMELEN) {
                                        had_error++;
                                        break;
                                }
                                bp += n;
                                buflen -= n;
                        }
                        break;
#else
                        host.h_name = bp;
                        if (_res.options & RES_USE_INET6) {
                                n = strlen(bp) + 1;     /* for the \0 */
                                if (n >= MAXHOSTNAMELEN) {
                                        had_error++;
                                        break;
                                }
                                bp += n;
                                buflen -= n;
                                map_v4v6_hostent(&host, &bp, &buflen);
                        }
                        h_errno = NETDB_SUCCESS;
                        return (&host);
#endif
                case T_A:
                case T_AAAA:
                        if (strcasecmp(host.h_name, bp) != 0) {
                                syslog(LOG_NOTICE|LOG_AUTH,
                                       AskedForGot, host.h_name, bp);
                                cp += n;
                                continue;       /* XXX - had_error++ ? */
                        }
                        if (n != host.h_length) {
                                cp += n;
                                continue;
                        }
                        if (!haveanswer) {
                                register int nn;

                                host.h_name = bp;
                                nn = strlen(bp) + 1;    /* for the \0 */
                                bp += nn;
                                buflen -= nn;
                        }

			buflen -= sizeof(align) - ((u_long)bp % sizeof(align));
                        bp += sizeof(align) - ((u_long)bp % sizeof(align));

                        if (bp + n >= &hostbuf[sizeof hostbuf]) {
                                dprintf("size (%d) too big\n", n);
                                had_error++;
                                continue;
                        }
                        if (hap >= &h_addr_ptrs[MAXADDRS-1]) {
                                if (!toobig++)
                                        dprintf("Too many addresses (%d)\n",
                                                MAXADDRS);
                                cp += n;
                                continue;
                        }
                        bcopy(cp, *hap++ = bp, n);
                         bp += n;
                        buflen -= n;
                        cp += n;
                        break;
                default:
                        abort();
                }
                if (!had_error)
                        haveanswer++;
        }
        if (haveanswer) {
                *ap = NULL;
                *hap = NULL;
# if defined(RESOLVSORT)
                /*
                 * Note: we sort even if host can take only one address
                 * in its return structures - should give it the "best"
                 * address in that case, not some random one
                 */
                if (_res.nsort && haveanswer > 1 && qtype == T_A)
                        addrsort(h_addr_ptrs, haveanswer);
# endif /*RESOLVSORT*/
                if (!host.h_name) {
                        n = strlen(qname) + 1;  /* for the \0 */
                        if (n > buflen || n >= MAXHOSTNAMELEN)
                                goto no_recovery;
                        strcpy(bp, qname);
                        host.h_name = bp;
                        bp += n;
                        buflen -= n;
                }
                if (_res.options & RES_USE_INET6)
                        map_v4v6_hostent(&host, &bp, &buflen);
                h_errno = NETDB_SUCCESS;
                return (&host);
        }
 no_recovery:
        h_errno = NO_RECOVERY;
        return (NULL);
}

#ifndef HAVE_RES_ISOURSERVER
static int
res_isourserver(inp)
        const struct sockaddr_in *inp;
{
        struct sockaddr_in ina;
        register int ns, ret;

        ina = *inp;
        ret = 0;
        for (ns = 0;  ns < _res.nscount;  ns++) {
                register const struct sockaddr_in *srv = &_res.nsaddr_list[ns];

                if (srv->sin_family == ina.sin_family &&
                    srv->sin_port == ina.sin_port &&
                    (srv->sin_addr.s_addr == INADDR_ANY ||
                     srv->sin_addr.s_addr == ina.sin_addr.s_addr)) {
                        ret++;
                        break;
                }
        }
        return (ret);
}

#endif

#ifndef HAVE_RES_QUERIESMATCH
static int
res_nameinquery(name, type, class, buf, eom)
        const char *name;
        register int type, class;
        const u_char *buf, *eom;
{
        register const u_char *cp = buf + HFIXEDSZ;
        int qdcount = ntohs(((HEADER*)buf)->qdcount);

        while (qdcount-- > 0) {
                char tname[MAXDNAME+1];
                register int n, ttype, tclass;

                n = dn_expand(buf, eom, cp, tname, sizeof tname);
                if (n < 0)
                        return (-1);
                cp += n;
                ttype = _getshort(cp); cp += INT16SZ;
                tclass = _getshort(cp); cp += INT16SZ;
                if (ttype == type &&
                    tclass == class &&
                    strcasecmp(tname, name) == 0)
                        return (1);
        }
        return (0);
}

static int
res_queriesmatch(buf1, eom1, buf2, eom2)
        const u_char *buf1, *eom1;
        const u_char *buf2, *eom2;
{
        register const u_char *cp = buf1 + HFIXEDSZ;
        int qdcount = ntohs(((HEADER*)buf1)->qdcount);

        if (qdcount != ntohs(((HEADER*)buf2)->qdcount))
                return (0);
        while (qdcount-- > 0) {
                char tname[MAXDNAME+1];
                register int n, ttype, tclass;

                n = dn_expand(buf1, eom1, cp, tname, sizeof tname);
                if (n < 0)
                        return (-1);
                cp += n;
                ttype = _getshort(cp);  cp += INT16SZ;
                tclass = _getshort(cp); cp += INT16SZ;
                if (!res_nameinquery(tname, ttype, tclass, buf2, eom2))
                        return (0);
        }
        return (1);
}
#endif
