/****************************************************************
 *
 * async_dns_from_named.c : asyncronous dns lookup routines.
 *
 * Copyright (c) 1998 Yutaka Oiwa.
 *
 ****************************************************************/

#include "config.h"
#include "async_dns.h"
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/nameser.h>
#include <resolv.h>
#include <errno.h>
#include <malloc.h>

#if PACKETSZ > 1024
#define MAXPACKET NS_PACKETSZ
#else
#define MAXPACKET 1024
#endif

#define MAXRETRY 4

#if defined(sun) && !defined(__svr4__)
typedef int int32_t;
#endif

#include "async_dns_from_named.c"

struct query_entry {
  struct query_entry *next;
  int id;           /* query id for user layer */
  int dns_id;       /* query id in DNS packet */
  char *name;       /* original argument */
  char *searchname; /* name currentry searching */
  int mode;         /* 0: direct 1: domain->direct 2: direct->domain */
  char *querybuf;   /* body of query */
  int querylen;     /* length of packet */
  int req_type;
  int req_size;
  int totaldoms;
  int domainnum;
  int retrynum;
  int direct;
  int direct_herrno;
  
  int ok[MAXNS];
  asyncdns_callback_t func;
  void *closure;
  void *timer_id;
  int timeout;

  int got_nodata;
  int got_servfail;
};

struct hosts_entry {
  struct hosts_entry *next;
  struct hostent h;
};

static struct hosts_entry hosts_head = { NULL, /* ... */ };
static struct query_entry q_head = { NULL, /* ... */ };
static int id = 0;

static int inited = 0;
static int sock;
static debuglevel = 1;

static asyncdns_regcallback_t a_regfunc;
static asyncdns_remcallback_t a_remfunc;
static asyncdns_tregcallback_t a_tregfunc;
static asyncdns_tremcallback_t a_tremfunc;

static int nscount;
static int srchcount;

static void *regfunc_id;

static void dumpdata(char *dat, int len) {
  int addr = 0, i;
  for(; len > 0; len -= 16, addr += 16) {
    printf("%08x: ", addr);
    for(i = 0; i < 16; i++) {
      if (i <= len)
	printf("%02x ", dat[addr+i] & 255);
      else
	printf("   ");
    }
    printf("  ");
    for(i = 0; i < 16; i++) {
      if (i > len)
	break;
      if (isprint(dat[addr+i] & 255))
	printf("%c", dat[addr+i] & 255);
      else 
	printf(".");
    }
    printf("\n");
  }
}

static int read_hosts_file(void) {
  struct hosts_entry *p = &hosts_head;
  struct hosts_entry *ent;
  char buf[512];
  unsigned long int addr;
  char *c, *addrp, *name, *names[16];
  int i, j;
  FILE *fp;

  char **res_addr_list, **res_names;
  char *res_addr;

  fp = fopen(_PATH_HOSTS, "r");
  if (!fp)
    return 0;
  
  while(fgets(buf, 512, fp) != NULL) {
    c = strpbrk(buf, "#\n");
    if (c)
      *c = 0;

    for(c = buf; *c && isspace(*c); c++)
      ;
    if (!*c) continue; /* null line */

    addrp = c;
    c = strpbrk(c, " \t");
    if (!c) continue;
    *c++ = 0;
    
    addr = inet_addr(addrp);
    if (addr == (unsigned long int) -1)
      continue;

    for(i = 0; i < 16; i++) {
      if (!c) break;
      while(*c && isspace(*c)) c++;
      if (!*c)
	break;
      names[i] = c;
      c = strpbrk(c, " \t");
      if (c)
	*c++ = 0;
    }

    if (!i)
      continue;

    ent = (struct hosts_entry *)malloc(sizeof (struct hosts_entry));
    if (!ent) {
      goto memerror;
    }
    ent->next = NULL;
    ent->h.h_addrtype = AF_INET;
    ent->h.h_length = INADDRSZ;

    res_addr_list = (char **)malloc(sizeof (char *) * 2);
    if (!res_addr_list) {
      free(ent);
      goto memerror;
    }
    res_addr = (char *)malloc(sizeof (unsigned long int));
    if (!res_addr) {
      free(res_addr_list);
      free(ent);
      goto memerror;
    }
    *(unsigned long int *)res_addr = addr;
    res_addr_list[0] = res_addr;
    res_addr_list[1] = NULL;
    ent->h.h_addr_list = res_addr_list;
    
    res_names = (char **)malloc(sizeof(char *) * i);
    if (!res_names) {
      free(res_addr);
      free(res_addr_list);
      free(ent);
      goto memerror;
    }
    
    name = strdup(names[0]);
    if (!name) {
      free(res_names);
      free(res_addr);
      free(res_addr_list);
      free(ent);
      goto memerror;
    }
    ent->h.h_name = name;

    for(j = 1; j < i; j++) {
      name = strdup(names[j]);
      if (!name) {
	for(j--; j >= 0; j--)
	  free(res_names[j]);
	free(res_names);
	free(res_addr);
	free(res_addr_list);
	free(ent);
	goto memerror;
      }
      res_names[j - 1] = name;
    }
    res_names[i - 1] = NULL;

    ent->h.h_aliases = res_names;

    p->next = ent;
    p = ent;

/*    {
      int j;
      printf("\naddr = %08x, num = %d\n", addr, i);
      for(j = 0; j < i; j++)
	printf("name[%d] = \"%s\"\n", j, names[j]);
    } */
  }
  return 1;

 memerror:
  errno = ENOMEM;
  return 0;
}

static struct hostent *search_static_byname(const char *name, int mode) {
  struct hosts_entry *e;
  struct hostent *p;
  int i;
  char **names;
  if (mode) {
    char buf[512];
    if (mode == 2) {
      p = search_static_byname(name, 0);
      if (p)
	return p;
    }
    for(i = 0; i < srchcount; i++) {
      if (strlen(name) + strlen(_res.dnsrch[i]) > 500) 
	continue;
      sprintf(buf, "%s.%s", name, _res.dnsrch[i]);
      p = search_static_byname(buf, 0);
      if (p)
	return p;
    }
    if (mode == 1) {
      return search_static_byname(name, 0);
    }
    return 0;
  }

  for(e = hosts_head.next; e; e = e->next) {
    if (strcasecmp(e->h.h_name, name) == 0)
      return &(e->h);
    for(names = e->h.h_aliases; *names; names++) {
      if(strcasecmp(*names, name) == 0)
	return &(e->h);
    }
  }

  return 0;
}

static struct hostent *search_static_byaddr(const char *addr, int size) {
  struct hosts_entry *e;
  char **addrs;

  for(e = hosts_head.next; e; e = e->next) {
    if (e->h.h_length == size && bcmp(e->h.h_addr, addr, size) == 0)
      return &(e->h);
  }

  return 0;
}

static void remove_entry(struct query_entry *q) {
  struct query_entry *p;
  for(p = &q_head; p->next; p = p->next) {
    if (p->next == q) {
      if (a_tremfunc && q->timer_id)
	a_tremfunc(q->timer_id);
      p->next = q->next;
      free(q->querybuf);
      free(q->searchname);
      free(q->name);
      free(q);
      return;
    }
  }
}

int asyncdns_abort_request(int id) {
  struct query_entry *p;
  for(p = &q_head; p->next; p = p->next) {
    struct query_entry *q = p->next;
    if (q->id == id) {
      if (a_tremfunc && q->timer_id)
	a_tremfunc(q->timer_id);
      p->next = q->next;
      free(q->querybuf);
      free(q->searchname);
      free(q->name);
      free(q);
      return 1;
    }
  }
  return 0;
}

static int send_query(struct query_entry *q)
{
  int succ = 0;
  int i;
  for(i = 0; i < _res.nscount; i++) {
    if (q->ok[i]){
      if (sendto(sock, q->querybuf, q->querylen, 0, 
		 (struct sockaddr *)&_res.nsaddr_list[i], sizeof (struct sockaddr_in)) != q->querylen) {
	q->ok[i] = 0;
      } else {
	succ++;
      }
    }
  }
  return succ;
}

static void asyncdns_timerelapsed(int id) {
  struct query_entry *q;
  int r;
  for(q = q_head.next; q; q = q->next) {
    if (q->id == id)
      break;
  }
  if (!q) {
    if (debuglevel >= 3)
      printf("id %d is not sent or already cancelled.\n", id);
    return;
  }

  q->timer_id = 0; /* Timer is cleared */

  if (q->retrynum != MAXRETRY - 1) {
    q->retrynum++;
    q->timeout = RES_TIMEOUT << q->retrynum;
    if (send_query(q)) {
      q->timer_id = a_tregfunc(asyncdns_timerelapsed, id, q->timeout);
      return;
    } else {
      h_errno = TRY_AGAIN;
    }
  }
  if (q->direct) {
    h_errno = TRY_AGAIN;
  } else if (q->direct_herrno != -1) {
    h_errno = q->direct_herrno;
  } else if (q->got_nodata) {
    h_errno = NO_DATA;
  } else if (q->got_servfail) {
    h_errno = TRY_AGAIN;
  }
  q->func(q->closure, q->id, NULL);
  remove_entry(q);
  return;
}

/* result: 0 = OK, -1 = ERROR, -2 = NO MORE */
static int make_next_query(struct query_entry *q)
{
  int dom, retry, i, n;
  char *target, *domain;
  if (q->domainnum == -1) { /* first time */
    q->direct_herrno = -1;
    q->totaldoms = (q->mode ? srchcount + 1 : 1);
    for(i = 0; i < nscount; i++) {
      q->ok[i] = 1;
    }
    q->got_nodata = 0;
    q->got_servfail = 0;
  }

  q->domainnum++;
  q->retrynum = 0;
  if (q->domainnum >= q->totaldoms) {
    /* no more */
    return -2;
  }
  dom = q->domainnum;
  if (q->mode == 0)
    dom = -1;
  else if (q->mode == 1) {
    if (dom == q->totaldoms - 1)
      dom = -1;
  } else
    dom--;
  
  q->direct = (dom == -1);
  if (dom == -1) {
    target = strdup(q->name);
  } else {
    domain = _res.dnsrch[dom];
    target = malloc(strlen(q->name) + strlen(domain) + 2);
    if (target)
      sprintf(target, "%s.%s", q->name, domain);
  }
  
  if (!target) {
    errno = ENOMEM;
    h_errno = NETDB_INTERNAL;
    return -1;
  }
  
  n = res_mkquery (QUERY, target, C_IN, q->req_type, NULL, 0, NULL, q->querybuf, MAXPACKET);
  
  if (n <= 0) {
    free(target);
    h_errno = NO_RECOVERY;
    return -1;
  }
  
  q->dns_id = ((HEADER *)q->querybuf)->id;
  if (q->searchname)
    free(q->searchname);
  q->searchname = target;
  q->querylen = n;
  q->timeout = RES_TIMEOUT;
  return 0;
}

static void asyncdns_dataarrived(int fd) {
  char buf[MAXPACKET];
  int n, dns_id;
  struct sockaddr_in addr;
  int alen = sizeof addr;
  HEADER *hp;
  struct query_entry *q;
  struct hostent *h;

  n = recvfrom(fd, buf, sizeof buf, 0, (struct sockaddr *)&addr, &alen);

  if (!res_isourserver(&addr)) {
    if (debuglevel >= 1)
      fprintf(stderr, "warning: wrong reply from %s\n", inet_ntoa(&addr));
    return;
  }
  if (debuglevel >= 3)
    printf("%d bytes data arrived\n", n);

  if (n < HFIXEDSZ) {
    if (debuglevel >= 1)
      fprintf(stderr, "bad sized (%d) message\n", n);
    return;
  }
  
  hp = (HEADER *) buf;
  dns_id = hp->id;
  
  for(q = q_head.next; q; q = q->next) {
    if (q->dns_id == dns_id)
      break;
  }
  if (!q) {
    if (debuglevel >= 3)
      printf("dns_id %d is not sent or already cancelled.\n", dns_id);
    return;
  }

  if (!res_queriesmatch(q->querybuf, q->querybuf + q->querylen,
			buf, buf + sizeof buf)) {
    if (debuglevel >= 3)
      printf("dns_id %d response does not match its query.\n", dns_id);
    return;
  }

  if (debuglevel >= 3)
    printf("id %d data arriving.\n", id);

  if (debuglevel >= 3)
    dumpdata(buf, n);

  h = NULL;
  switch (hp->rcode) {
  case NXDOMAIN:
    h_errno = HOST_NOT_FOUND;
    break;
  case SERVFAIL:
    h_errno = TRY_AGAIN;
    break;
  case NOERROR:
    if (ntohs(hp->ancount) == 0) {
      h_errno = NO_DATA;
      break;
    }
    host.h_length = q->req_size;
    host.h_addrtype = q->req_type;
    h = getanswer((querybuf *) buf, n, q->searchname, q->req_type);
    if (debuglevel >= 3) {
      if (h)
	printf("positive reply (%d)\n", h);
      else 
	printf("negative reply (%d)\n", h_errno);
    }
    q->func(q->closure, q->id, h);
    remove_entry(q);
    return;

  case FORMERR:
  case NOTIMP:
  case REFUSED:
  default:
    h_errno = NO_RECOVERY;
    break;
  }

  if (q->direct)
    q->direct_herrno = h_errno;
  if (h_errno != NO_RECOVERY) {
    if (h_errno == NO_DATA)
      q->got_nodata++;
    if (h_errno == TRY_AGAIN) {
      q->got_servfail++;
    }
    if (make_next_query(q) == 0) {
      if (a_tremfunc)
	a_tremfunc(q->timer_id);
      q->timer_id = 0;
      
      if (send_query(q)) {
	if (a_tregfunc)
	  q->timer_id = a_tregfunc(asyncdns_timerelapsed, q->id, q->timeout);
	return;
      }
    }
  }
  if (q->direct_herrno != -1)
    h_errno = q->direct_herrno;
  else if (q->got_nodata)
    h_errno = NO_DATA;
  else if (q->got_servfail)
    h_errno = TRY_AGAIN;
  
  if (debuglevel >= 3)
    printf("negative reply (%d)\n", h_errno);
  q->func(q->closure, q->id, NULL);
  remove_entry(q);
}

static int register_entry(asyncdns_callback_t f, void *closure, char *nname,
			  int type, int mode) {
  /* nname must be dynamically allocated string */
  struct query_entry *q;
  char *buf;
  int n;
  int i;
  int succ;

  if (srchcount == 0)
    mode = 0; /* if no domain is available, only we can do is direct search */

  q = (struct query_entry *)malloc(sizeof (struct query_entry));
  if (!q) {
    errno = ENOMEM;
    h_errno = NETDB_INTERNAL;
    return 0;
  }
  if (!q) {
    errno = ENOMEM;
    h_errno = NETDB_INTERNAL;
    free(nname);
    return -1;
  }
  buf = malloc(MAXPACKET);
  if (!buf) {
    free (q);
    errno = ENOMEM;
    h_errno = NETDB_INTERNAL;
    return -1;
  }
  
  q->next = q_head.next;
  q->id = ++id;
  q->name = nname;
  q->searchname = NULL;
  q->mode = mode;
  q->querybuf = buf;
  q->req_type = type;
  q->req_size = INADDRSZ;
  q->func = f;
  q->closure = closure;
  q->timer_id = NULL;
  q->domainnum = -1;

  if (make_next_query(q) < 0) {
    free(q);
    free(nname);
    free(buf);
    h_errno = NO_RECOVERY;
    return -1;
  }

  if (!send_query(q)) {
    perror("asyncdns_gethostbyname: sendto");
    free(q);
    free(nname);
    free(buf);
    h_errno = TRY_AGAIN;
    return -1;
  }

  if (a_tregfunc)
    q->timer_id = a_tregfunc(asyncdns_timerelapsed, q->id, q->timeout);

  q_head.next = q;

  return q->id;
}

int asyncdns_gethostbyname(asyncdns_callback_t f, void *closure,
			   struct hostent **r, char *name) {
  char *nname;
  static unsigned long int a;
  int mode;

  if (!inited) {
    errno = EINVAL;
    h_errno = NETDB_INTERNAL;
    return -1;
  }

  a = inet_addr(name);
  if (a != (unsigned long int)-1) {
    host.h_name = name;
    host.h_aliases = host_aliases;
    host_aliases[0] = NULL;
    h_addr_ptrs[0] = (char *)&a;
    h_addr_ptrs[1] = NULL;
    host.h_addr_list = h_addr_ptrs;
    h_errno = NETDB_SUCCESS;
    *r = &host;
    return 0; /* immediate success */
  }

  if (*name == '\0') {
    h_errno = NO_DATA;
    return -1;
  }

  if (name[strlen(name) - 1] == '.') {
    name[strlen(name) - 1] = '\0';
    mode = 0;
  } else if (strchr(name, '.')) {
    mode = 2;
  } else {
    mode = 1;
  }
  
  {
    struct hostent *e = search_static_byname(name, mode);
    if (e) {
      *r = e;
      return 0;
    }
  }

  nname = strdup(name);
  if (!nname) {
    errno = ENOMEM;
    h_errno = NETDB_INTERNAL;
    return -1;
  }
  return register_entry(f, closure, nname, T_A, mode);
}

int asyncdns_gethostbyaddr(asyncdns_callback_t f, void *closure,
			   struct hostent **r,
			   const char *addr, int len, int type) {
  char *nname;
  struct hostent *p;

  if (!inited) {
    errno = EINVAL;
    h_errno = NETDB_INTERNAL;
    return -1;
  }
  if (type != AF_INET) {
    errno = EAFNOSUPPORT;
    h_errno = NETDB_INTERNAL;
    return -1;
  }
  if (len != INADDRSZ) {
    errno = EINVAL;
    h_errno = NETDB_INTERNAL;
    return -1;
  }

  p = search_static_byaddr(addr, len);
  if (p) {
    *r = p;
    return 0;
  }

  nname = malloc(50); /* enough for "255.255.255.255.in-addr.arpa" */
  if (!nname) {
    errno = ENOMEM;
    h_errno = NETDB_INTERNAL;
    return -1;
  }
  sprintf(nname, "%u.%u.%u.%u.in-addr.arpa", addr[3] & 255, addr[2] & 255, 
	  addr[1] & 255, addr[0] & 255);
  return register_entry(f, closure, nname, T_PTR, 0);
}

int asyncdns_set_debug_level(int d) {
  int o = debuglevel;
  debuglevel = d;
  return o;
}

int asyncdns_init(asyncdns_regcallback_t regfunc,
		  asyncdns_remcallback_t remfunc,
		  asyncdns_tregcallback_t tregfunc,
		  asyncdns_tremcallback_t tremfunc) {
  if (inited)
    return 1;
  if (!regfunc || !remfunc) {
    h_errno = NETDB_INTERNAL;
    errno = EINVAL;
    return 0;
  }
  if (res_init()) {
    if (debuglevel >= 1)
      fprintf(stderr, "asyncdns: resolver initialize failed");
    h_errno = NETDB_INTERNAL;
    return 0;
  }
  sock = socket(AF_INET, SOCK_DGRAM, 0);
  if (!sock) {
    if (debuglevel >= 1)
      fprintf(stderr, "asyncdns: cannot make socket");
    h_errno = NETDB_INTERNAL;
    return 0;
  }
  nscount = _res.nscount;
  srchcount = 0;
  while(_res.dnsrch[srchcount])
    srchcount++;
  
  a_regfunc = regfunc;
  a_remfunc = remfunc;
  a_tregfunc = tregfunc;
  a_tremfunc = tremfunc;
  regfunc_id = regfunc(asyncdns_dataarrived, sock);

  read_hosts_file();

  inited = 1;
  return 1;
}

