/****************************************************************
 * ItalkPipeConnection.cc : Italk-Server connection class
 *
 * (c) 1997-1998 Yutaka Oiwa.
 *
 * You may distribute this file under the terms of license
 * agreement specified in the files README and LICENCE.
 *
 * $Id: ItalkPipeConnection.cc,v 1.27 2005/06/01 13:57:55 yutaka Exp $
 ****************************************************************/

#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <errno.h>

#include "Defines.h"

#ifdef ENABLE_ASYNC_DNS
#include "asyncdns/async_dns_X.h"
#endif

static const char *short_strerror(int err, int h_err = 0)
{
  switch(err) {
  case -1:
    switch (h_err) {
    case NETDB_INTERNAL:
      return "DNS Error(INT).";
    case NO_RECOVERY:
      return "DNS Error(NR).";
    case TRY_AGAIN:
      return "DNS Fail.";
    case HOST_NOT_FOUND:
      return "Host not exist.";
    case NO_DATA:
      return "Addr. not exist.";
    default:
      return "DNS Error(-).";
    }
  case EISCONN: //
    return "Already Conn'ed.";
  case EINPROGRESS: //
    return "Connecting.";
  case ECONNREFUSED:
    return "Refused.";
  case EHOSTDOWN:
    return "Host Down.";
  case EHOSTUNREACH:
    return "Host Unreachable.";
  case ENETDOWN:
    return "Net Down.";
  case ENETUNREACH:
    return "Net Unreachable.";
  default:
    return strerror(err);
  }
}

static int create_socket(void) {
  int fd = socket(AF_INET, SOCK_STREAM, 0);
  if (fd < 0) {
    Dperror(16, "error: create_socket: socket()");
    exit(1);
  }
  int f = 1;
  setsockopt(fd, IPPROTO_TCP, TCP_NODELAY, (char *)&f, sizeof(f));
  f = 1;
  setsockopt(fd, SOL_SOCKET, SO_KEEPALIVE, (char *)&f, sizeof(f));
#ifndef NO_ASYNC_CONNECT
  f = fcntl(fd, F_GETFL, 0);
  f |= O_NONBLOCK;
  fcntl(fd, F_SETFL, f);
#endif
  return fd;
}

static char *parse_hostname(const char *host, int *port)
{
  char *p = NewString(host);
  char *c = strchr(p, ':');
  if (c) {
    *c = '\0';
    c++;
    struct servent *s = getservbyname(c, "TCP");
    if (s) {
      *port = ntohs(s->s_port);
    } else if (atoi(c)){
      *port = atoi(c);
    }
  }
  return p;
}

static int connect_socket(int s, unsigned long int a, int p)
{
  struct sockaddr_in cli;

  cli.sin_family = AF_INET;
  cli.sin_addr.s_addr = a;
  cli.sin_port = htons(p);

  return connect(s, (struct sockaddr*)&cli, sizeof(cli));
}

static char *pretty_hostname(const char *h)
{
  char *p = NewString(h);
  char *q = strchr(p, '.');
  if (q && inet_addr(p) == (unsigned)-1) {
    *q = '\0';
  }
  return p;
}

// Following Routine require X routines

#include <stdio.h>
#include <unistd.h>
#include <X11/Intrinsic.h>
#include "ItalkServer.h"
#include "LineBuffer.h"
#include "ItalkPipeConnection.h"

ItalkPipeConnection::ItalkPipeConnection(XtAppContext app_,
					 char *h, 
					 int waittime_,
					 int keepalivetime_,
					 ItalkServer_callback_t notify,
					 void *app_data) :
					 lbuf(cb_line_arrived, this)
{
#ifdef ENABLE_ASYNC_DNS
  asyncdns_X_init(app_);
#ifdef DEBUG
  if (Debug_Level & 64)
    asyncdns_set_debug_level(3);
  else
#endif
    asyncdns_set_debug_level(0);
#endif
  app = app_;
  port = 12345;
  soc = -1;
  waittime = waittime_;
  keepalivetime = keepalivetime_;
  num = 0; name[0] = 0; where[0] = 0; stat[0] = 0;
  protocol = 0;
  update_allowed = false;
  in_italk_section = in_user_section = false;
  status = INIT;
  input_id = 0;
  interval_id = 0;
  allow_fallback = true;

  fullname = parse_hostname(h, &port);
  
  char *p = pretty_hostname(fullname);
  char buf[256];
  sprintf(buf, "%.240s:%d", fullname, port);
  ser = new ItalkServer(p, port, buf, notify, app_data);
  delete [] p;
  make_connection();
}

void name_resolved(void *closure, int, struct hostent *ent)
{
  ItalkPipeConnection *self = (ItalkPipeConnection *)closure;
  if (ent)
    self->connect_to_server(((struct in_addr*)(ent->h_addr))->s_addr);
  else
    self->connect_to_server(INADDR_ANY); // error status
}

void ItalkPipeConnection::make_connection()
{
  unsigned long int host_addr;
  
  host_addr = inet_addr(fullname);
  if (host_addr == (unsigned long int)-1) {
    host_addr = INADDR_ANY; // paranoia
#ifdef ENABLE_ASYNC_DNS
    ser->DataStart();
    ser->AddMessage("Searching...");
    ser->DataFinished();
    struct hostent *ent = 0;
    int id = asyncdns_gethostbyname(name_resolved, this, &ent, fullname);
    if (id > 0) {
      set_state(NAME_WAIT);
      return;
    }
    else if (id == 0)
      host_addr = ((struct in_addr*)(ent->h_addr))->s_addr;
#else
    struct hostent *ent = gethostbyname(fullname);
    if (ent) 
      host_addr = ((struct in_addr*)(ent->h_addr))->s_addr;
#endif
  }
  connect_to_server(host_addr);
}

void ItalkPipeConnection::connect_to_server(unsigned long int addr)
{
  if (addr == INADDR_ANY) {
    // hostname error
    connect_failed(-1);
    return;
  }
  Dprintf(16, "make_connection %s, (%d->", fullname, soc);
  if (soc == -1)
    soc = create_socket();
  Dprintf(16, "%d,%s,%lx,%d)\n", soc, fullname, addr, port);
  
  int ret = connect_socket(soc, addr, port);
  if (ret < 0) {
    if (errno == EINPROGRESS) {
      ser->DataStart();
      ser->AddMessage("Connecting...");
      ser->DataFinished();
      set_state(CONN_WAIT);
    } else {
      Dperror(16, "ItalkPipeConnection::connect_socket: connect()");
      connect_failed(errno);
    }
  } else {
    connect_established();
  }
}

void ItalkPipeConnection::connect_established()
{
  int r = 1;
  setsockopt(soc, SOL_SOCKET, SO_REUSEADDR, (char *)&r, sizeof r);

#ifdef NO_ASYNC_CONNECT
  int f;
  // Nonblock mode setting if postponed to here.
  f = fcntl(soc, F_GETFL, 0);
  f |= O_NONBLOCK;
  fcntl(soc, F_SETFL, f);
#endif

  ser->DataStart();
  ser->AddMessage("Connected...");
  ser->DataFinished();

  // Initialize Connection-based parameter
  protocol = 0;
  update_allowed = 0;
  in_italk_section = in_user_section = false;
  set_state(CONN);
}

void ItalkPipeConnection::connect_failed(int err)
{
  const char *mes = short_strerror(err, h_errno);
  // failed
  close(soc);
  soc = -1;
  ser->DataStart();
  ser->AddMessage(mes);
  ser->DataFinished();
  set_state(FAILED);
}

void cb_X_time_elapsed(void *s, XtIntervalId *)
{
  ItalkPipeConnection *self = (ItalkPipeConnection *)s;
  Dprintf(16, "elapsed %s\n", self->fullname);
  self->interval_id = 0;
  self->make_connection();
}

void cb_X_keepalive_elapsed(void *s, XtIntervalId *)
{
  ItalkPipeConnection *self = (ItalkPipeConnection *)s;
  Dprintf(16, "keepalive %s\n", self->fullname);
  write(self->soc, "/\n", 2);
  self->keepalive_id = 0;
  self->keepalive_id = XtAppAddTimeOut(self->app, self->keepalivetime * 1000, 
				       cb_X_keepalive_elapsed, (void *)self);
}

void cb_X_connected(void *s, int *fdp, XtInputId*)
{
  ItalkPipeConnection *self = (ItalkPipeConnection *)s;

  Dprintf(16, "connected %s\n", self->fullname);
  int e1, e2 = 0;
  socklen_t l = sizeof e2;
  e1 = getsockopt(*fdp, SOL_SOCKET, SO_ERROR, (char *)&e2, &l);
  Dprintf(16, "getsockopt: len=%d, val=%d, status=%d\n", l, e2, e1);
  if (e1 == -1)
    e2 = errno;
  if (e2) {
    Dperror(16, "cb_X_connected: connect()");
    self->connect_failed(e2);
  } else {
    self->connect_established();
  }
}

void cb_X_data_arrived(void *s, int *fdp, XtInputId*)
{
  ItalkPipeConnection *self = (ItalkPipeConnection *)s;
  char buf[512];
  int i = read(*fdp, buf, 256);
  Dprintf(16, "arrive %s (%d)\n", self->fullname, i);
  if (i < 0) {
    if (errno == EWOULDBLOCK)
	return;
    Dperror(16, "cb_X_data_arrived");
    i = -1;
  }
  self->lbuf.put(buf, i);
}

void ItalkPipeConnection::set_state(status_t s)
{
  Dprintf(16, "setstate %s (%d->%d)\n", fullname, status, s);
  switch (status) {
  case INIT:
  case HALT:
  case NAME_WAIT:
    break;
  case CONN_WAIT:
    XtRemoveInput(input_id);
    input_id = 0;
    break;
  case CONN:
  case WRONGPORT_1:
    XtRemoveInput(input_id);
    input_id = 0;
    if (keepalive_id)
      XtRemoveTimeOut(keepalive_id);
    keepalive_id = 0;
    break;
  case FAILED:
    if (interval_id)
      XtRemoveTimeOut(interval_id);
    interval_id = 0;
    break;
  }
  status = s;
  switch (status) {
  case INIT:
    abort();
  case NAME_WAIT:
    break;
  case CONN_WAIT:
    input_id = XtAppAddInput(app, soc, (XtPointer)XtInputWriteMask,
			     cb_X_connected, (void *)this);
    break;
  case CONN:
  case WRONGPORT_1:
    input_id = XtAppAddInput(app, soc, (XtPointer)XtInputReadMask,
			     cb_X_data_arrived, (void *)this);
    break;
  case FAILED:
    interval_id = XtAppAddTimeOut(app, waittime * 1000, cb_X_time_elapsed, (void *)this);
    break;
  case HALT:
    break;
  }
}

inline char *remove_newline(char *s)
{
  int n=strlen(s);
  if (n>0 && s[n-1]=='\n') s[n-- -1]='\0';
  if (n>0 && s[n-1]=='\r') s[n-- -1]='\0';
  return s;
}

void cb_line_arrived(void *s, char *buf, int len)
{
  ItalkPipeConnection *self = (ItalkPipeConnection *)s;
  
  if (self->status == ItalkPipeConnection::WRONGPORT_1)
    self->wrongport_process(buf, len);
  else
    self->getline(buf, len);
}

void ItalkPipeConnection::wrongport_process(char *buf, int len)
{
  if (len <= -1) { // EOF
    Dprintf(16, "DATAH: EOF\n");
    close(soc);
    soc = -1;
    if (allow_fallback) {
      fprintf(stderr,
	      "xitalkbiff: Server \"%.200s:%d\" is for chat only. \n"
	      "            Trying \"%.200s:%d\".\n",
	      fullname, port, fullname, port-1);
      
      port--;
      this->make_connection();
    } else {
      set_state(HALT);
    }
    return;
  }
  remove_newline(buf);
  Dprintf(16, "DATAH: %s\n", buf);
  if (strcmp(buf, "# /? for help.") == 0) { // italk-
    allow_fallback = false;
    fprintf(stderr,
	    "xitalkbiff: Server \"%.200s:%d\" is for chat only, \n"
	    "            and fallback is not supported. aborting.\n",
	    fullname, port);
    sprintf(buf, "\033[43;31mUnsupported Server.\033[m");
    ser->DataStart();
    ser->AddMessage(buf, 1); // urgent
    ser->DataFinished();
    write(soc, "/q\n", 3);
  }
}

void ItalkPipeConnection::getline(char *buf, int len)
{
  if (len <= -1) { // EOF
    close(soc);
    soc = -1;
    ser->DataStart();
    if (len == -1)
      ser->AddMessage("Conn. Closed.");
    else
      ser->AddMessage(short_strerror(errno));
    ser->DataFinished();
    set_state(FAILED);
    return;
  }
  remove_newline(buf);
  Dprintf(32, "DATA: %s\n", buf);
  if (protocol >= 100) {
    if (strncmp(buf, "#! ", 3) == 0) {
      buf += 3;
    } else {
      return;
    }
  }

  if (strncmp(buf, "# What's", 8) == 0 ||
      strncmp(buf, "# \xa4\xaa\xcc\xbe\xc1\xb0\xa4\xf2\xa4\xc9\xa4\xa6\xa4\xbe", 16) == 0) {
    // connected to italk main port (^^;
    // In case of server with ItP >1.0, this check is passed by above #! check.
    ser->DataStart();
    ser->AddMessage("Old Chat Port.");
    ser->DataFinished();
    write(soc, "/q\n", 3); // \cD\n
    set_state(WRONGPORT_1);
    return;
  } else if (strncmp(buf, "# Italk Protocol ", 17) == 0) {
    int i, j;
    sscanf(buf + 17, "%d.%d", &i, &j);
    protocol = i * 100 + j;
    Dprintf(32, "protocol=%d\n", protocol);
    if (protocol >= 100)
      write(soc, "/x type=biff\r\n/wa\r\n", 19);
  } else if (strcmp(buf, "<italk>")==0
#ifdef OLD_COMPAT
	     || strncmp(buf, "## __", 5) == 0
#endif
	     ) {
    if (protocol == 0) {
      protocol = (buf[0] == '#' ? 1 : 2);
      Dprintf(32, "protocol=%d\n", protocol);
    }
    in_italk_section = 1;
    ser->DataStart();
    Dprintf(32, "Start!\n");
  } else if (in_italk_section
	     && (strcmp(buf, "</italk>")==0
#ifdef OLD_COMPAT
		 || strncmp(buf, "## --", 5) == 0
#endif
		 )) {
    Dprintf(32, "Done.\n");
    in_italk_section = 0;
    if (protocol >= 100)
      update_allowed = 1;
    ser->DataFinished();
    if (keepalivetime >= 1) {
	Dprintf(16, "start keepalive %s\n", fullname);
	keepalive_id = 0;
	keepalive_id = XtAppAddTimeOut(app, keepalivetime * 1000, 
					     cb_X_keepalive_elapsed, (void *)this);
    }
  } else if (protocol >= 2
	     && ((in_italk_section && strcmp(buf, "<user>") == 0)
		 || (update_allowed && strcmp(buf, "<newuser>") == 0))) {
    in_user_section = 1;
    num = 0;
    name[0] = 0;
    where[0] = 0;
    stat[0] = 0;
  } else if (in_user_section && strncmp(buf, "userno=", 7) == 0) {
    num = atoi(buf+7);
  } else if (in_user_section && strncmp(buf, "handle=", 7) == 0) {
    strncpy(name, buf + 7, 80);
    name[79] = 0;
  } else if (in_user_section && strncmp(buf, "host=", 5) == 0) {
    strncpy(where, buf + 5, 80);
    where[79] = 0;
  } else if (in_user_section && strncmp(buf, "status=", 7) == 0) {
    strncpy(stat, buf + 7, 80);
    stat[79] = 0;
  } else if (in_user_section && (strcmp(buf, "</user>") == 0
				 || strcmp(buf, "</newuser>") == 0)) {
    Dprintf(32, "AddMonkeyNew %s (%d,%s,%s,%s,%p)\n",fullname,num,name,where,stat,ser);
    ser->AddMonkey(new Monkey(num,name,where,stat,ser));
    in_user_section = 0;
  } else if (update_allowed && strncmp(buf, "newhandle=", 10) == 0) {
    char *p = strchr(buf + 10, ',');
    if (p) {
      int no = atoi(buf + 10);
      Dprintf(32, "UpdateHandle (%d,%s)\n", no, p + 1);
      ser->UpdateHandle(no, p + 1);
    }
  } else if (update_allowed && strncmp(buf, "newstatus=", 10) == 0) {
    char *p = strchr(buf + 10, ',');
    if (p) {
      int no = atoi(buf + 10);
      Dprintf(32, "UpdateStatus (%d,%s)\n", no, p + 1);
      ser->UpdateStatus(no, p + 1);
    }
  } else if (update_allowed && strncmp(buf, "logout=", 7) == 0) {
    int no = atoi(buf + 7);
    Dprintf(32, "Logout (%d).\n", no);
    ser->LogoutMonkey(no);
  } else if (update_allowed && strncmp(buf, "disconnect=", 11) == 0) {
    int no = atoi(buf + 11);
    Dprintf(32, "Disconnect (%d).\n", no);
    ser->LogoutMonkey(no);
  }
#ifdef OLD_COMPAT
  else if (protocol == 1 && in_italk_section
	   && ('0' <= buf[0] && buf[0] <= '9')) {
    int num=atoi(buf);
    char *name=strchr(buf, ' ');
    if (!name) return;
    if (strlen(name) < 30) return;
    name++;
    name[28] = 0;
    char *where = "";
    char *stat = "";
    for (int j = 27; j > 0 ; j--)
      if (name[j] == '@') {
	name[j] = 0;
	where = &name[j+1];
	break;
      }
    if (name[29] == '<') {
      stat = &name[30];
      char *p = strrchr(stat, '>');
      if (p)
	*(p) = 0;
    }
    Dprintf(32, "AddMonkeyOld(%s,%d,%s,%s,%s,%p)\n",fullname, num,name,where,stat,ser);
    ser->AddMonkey(new Monkey(num, name, where, stat, ser));
  }
#endif
}

ItalkPipeConnection::~ItalkPipeConnection(void)
{
  if (soc >= 0) {
    if (protocol >= 100) {
      Dprintf(32, "sending /q.\n");
      write(soc, "/q\r\n", 4);
      close(soc);
    }
  }
}

